﻿/*
 * Unite 2013: Custom Build Processes with Unity3D
 * =======================================================
 * Author: William Roberts - http://www.williamroberts.net
 * Date:   August 27, 2013
 * 
 * Schell Games - http://www.schellgames.com
 */

using System.Collections.Generic;
using System.IO;
using System;

namespace UniteExample.MSBuild.Extensions.Utility
{
    public static class ProjectUtility
    {
        private readonly static string[] _baseStructure;

        static ProjectUtility()
        {
            _baseStructure = new string[]
            {
                "Assets",
                "ProjectSettings"
            }; 
        }

        /// <summary>
        /// If the specified path is an Unity3D project, it will be returned as the result. If the path is not a Unity project,
        /// the immedite children of the directory will be inspected. 
        /// </summary>
        /// <param name="rootDirectory">The path to inspect for Unity3D projects.</param>
        /// <returns>
        /// An array containing zero or more paths.
        /// </returns>
        public static DirectoryInfo[] GetUnityProjects(DirectoryInfo rootDirectory)
        {
            List<DirectoryInfo> actualProjects = new List<DirectoryInfo>();

            if (rootDirectory == null)
                throw new NullReferenceException("The root directory must not be an null value!");

            if (rootDirectory.Exists)
            {
                if (IsUnityProject(rootDirectory.FullName))
                {
                    actualProjects.Add(rootDirectory);
                }
                else
                {
                    DirectoryInfo[] possibleProjects = rootDirectory.GetDirectories();

                    for (int i = 0; i < possibleProjects.Length; i++)
                    {
                        if (IsUnityProject(possibleProjects[i]))
                            actualProjects.Add(possibleProjects[i]);
                    }
                }
            }

            return actualProjects.ToArray();
        }

        /// <summary>
        /// Determines if the specified path is an Unity3D project.
        /// </summary>
        /// <param name="path">Path to the directory to test.</param>
        /// <returns>True if the path is an Unity 3D Project.</returns>
        public static bool IsUnityProject(string path)
        {
            return IsUnityProject(new DirectoryInfo(path));
        }

        /// <summary>
        /// Determines if the specified path is an Unity3D project.
        /// </summary>
        /// <param name="path">Path to the directory to test.</param>
        /// <returns>True if the path is an Unity 3D Project.</returns>
        public static bool IsUnityProject(DirectoryInfo projectDirectory)
        {
            if (projectDirectory == null  || !projectDirectory.Exists)
                return false;

            for(int i=0; i < _baseStructure.Length; i++)
            {
                string subDir = Path.Combine(projectDirectory.FullName, _baseStructure[i]);

                if (!Directory.Exists(subDir))
                    return false;
            }

            return true;
        }
    }
}
