﻿/*
 * Unite 2013: Custom Build Processes with Unity3D
 * =======================================================
 * Author: William Roberts - http://www.williamroberts.net
 * Date:   August 27, 2013
 * 
 * Schell Games - http://www.schellgames.com
 */

using System;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Net.Sockets;
using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using UniteExample.MSBuild.Extensions.Utility;

namespace UniteExample.MSBuild.Extensions.Tasks
{
    /// <summary>
    /// Launches the Unity Editor in batch mode and executes the specifed method.
    /// </summary>
    public class UnityTask : Task
    {
        private const int PortNumber = 21000;

        /// <summary>
        /// Path of the Unity Editor executable. Note that both the Windows and OSX Editors are valid.
        /// </summary>
        [Required]
        public string UnityExecutable { get; set; }

        /// <summary>
        /// Path to the Unity project(s) that should be built. The path is allowed to either be
        /// an Unity project or an parent directory that contains many Unity projects.
        /// </summary>
        [Required]
        public string ProjectPath { get; set; }

        /// <summary>
        /// A static parameterless method that should be executed immediately after each Unity project
        /// loads.
        /// </summary>
        [Required]
        public string MethodToExecute { get; set; }

        /// <summary>
        /// Defines the platform that should be built.
        /// </summary>
        public string BuildTarget { get; set; }


        public override bool Execute()
        {
            DirectoryInfo pathToTest = new DirectoryInfo(ProjectPath);
            DirectoryInfo[] projectPaths;

            if (!pathToTest.Exists)
            {
                Log.LogError("The specified project directory '{0}' does not exist!", pathToTest.FullName);
                return false;
            }

            if (!File.Exists(UnityExecutable))
            {
                Log.LogError("The specified Unity executable '{0}' could not be located!", UnityExecutable);
                return false;
            }

            if (string.IsNullOrEmpty(MethodToExecute))
            {
                Log.LogError("You must specify an method for the Unity Editor to execute!");
                return false;
            }

            // Searches the path for Unity projects. If the path is a Unity project, it will be returned.
            // Otherwise, it will be searched for Unity projects. Note that the search only searches the
            // immediate children of the parent directory.
            projectPaths = ProjectUtility.GetUnityProjects(pathToTest);

            foreach (DirectoryInfo path in projectPaths)
            {
                using (Process unityProcess = new Process())
                {
                    unityProcess.StartInfo.FileName = UnityExecutable;
                    unityProcess.StartInfo.Arguments = MakeArguments(path.FullName, MethodToExecute, BuildTarget);
                    unityProcess.Start();

                    RunNetworkServer(unityProcess, PortNumber);

                    unityProcess.WaitForExit();

                    if (unityProcess.ExitCode != 0)
                    {
                        Log.LogError("Unity returned exit code '{0}'", unityProcess.ExitCode);
                        return false;
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// Listens for a socket connection from the Unity Editor. Once an connection is made,
        /// the function while block while receiving and logging messages from the client.
        /// </summary>
        /// <param name="unityProcess">The process that will be remotely connecting.</param>
        /// <param name="portNumber">The port number that should be monitored for connections.</param>
        /// <returns>True on success, false on failure.</returns>
        private bool RunNetworkServer(Process unityProcess, int portNumber)
        {
            TcpListener server;
            TcpClient client = null;

            server = new TcpListener(IPAddress.Loopback, portNumber);
            server.Start();

            Log.LogMessage("Waiting for the Unity Editor process to connect...");

            try
            {
                do
                {
                    // Stop waiting for connection if the child Unity Editor process died.
                    if (unityProcess.HasExited)
                        return false;

                    // Wait for the Unity Editor to connect to the editor. We only care about one connection,
                    // so stop listening for more clients once we have one.
                    if (server.Pending())
                    {
                        client = server.AcceptTcpClient();
                        server.Stop();
                    }
                }
                while (client == null);

                Log.LogMessage("Client Connected!");

                // Start reading and logging data sent from the Unity Editor. 
                using (BinaryReader reader = new BinaryReader(client.GetStream()))
                {
                    try
                    {
                        while (true)
                        {
                            string msgFromUnity = reader.ReadString();

                            // TODO: Perform fancier handling of messages here. You could potentially send XML or JSON
                            //       data over the socket. This would allow complex data to be sent back to the build tool.

                            Log.LogMessage(msgFromUnity);
                        }
                    }
                    catch (Exception)
                    {
                        // This typically occurs once the client closes the connection.
                    }
                }
            }
            finally
            {
                if (client != null)
                {
                    client.Close();
                    client = null;
                }
            }

            return true;
        }

        /// <summary>
        /// Formats the command-line argument string that will be passed to the Unity Editor.
        /// </summary>
        /// <param name="projectPath">Full path to the Unity project to load.</param>
        /// <param name="method">Static parameterless method to execute after the project loads.</param>
        /// <param name="buildTarget">The type of build that should be created.</param>
        /// <returns>A command-line argument string ready for use by the Process object.</returns>
        private string MakeArguments(string projectPath, string method, string buildTarget)
        {
            // Note: The -buildTarget flag is not a standard Unity flag. It was made up to demonstrate the ability to send
            //       custom arguments to the Editor.
            return string.Format(
                @"-projectPath  ""{0}"" -executeMethod {1} -quit -batchmode -nographics -buildTarget {2}",
                projectPath,
                method,
                buildTarget
            );
        }
    }
}
