﻿/*
 * Unite 2013: Custom Build Processes with Unity3D
 * =======================================================
 * Author: William Roberts - http://www.williamroberts.net
 * Date:   August 27, 2013
 * 
 * Schell Games - http://www.schellgames.com
 */

using System.IO;
using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using UniteExample.MSBuild.Extensions.Utility;

namespace UniteExample.MSBuild.Extensions.Tasks
{
    /// <summary>
    /// Copies the specified assembliy to one or more Unity projects.
    /// </summary>
    public class CopyTask : Task
    {
        private const string AssemblyOutputDirectory = "Lib";

        /// <summary>
        /// The path to the assembly that should be copied to the Unity project(s).
        /// Typically this will be the $(TargetPath) property in the MSBuild script.
        /// </summary>
        [Required]
        public string TargetAssembly { get; set; }

        /// <summary>
        /// Path to the Unity project(s) that the assembly should be copied to. The path is allowed
        /// to either be an Unity project or an parent directory that contains many Unity projects.
        /// </summary>
        [Required]
        public string UnityProjectPath { get; set; }


        public override bool Execute()
        {
            FileInfo targetAssemblyInfo = new FileInfo(TargetAssembly);
            DirectoryInfo projectPathInfo = new DirectoryInfo(UnityProjectPath);
            DirectoryInfo[] projects;

            if (!targetAssemblyInfo.Exists)
            {
                Log.LogError("Could not find an assembly to copy at: {0}", targetAssemblyInfo.FullName);
                return false;
            }

            // Searches the path for Unity projects. If the path is a Unity project, it will be returned.
            // Otherwise, it will be searched for Unity projects. Note that the search only searches the
            // immediate children of the parent directory.
            projects = ProjectUtility.GetUnityProjects(projectPathInfo);

            if (projects.Length == 0)
            {
                Log.LogError("Could not find any Unity3D projects at: {0}", projectPathInfo.FullName);
                return false;
            }

            foreach (DirectoryInfo projectInfo in projects)
            {
                Copy(targetAssemblyInfo, projectInfo);
            }

            return true;
        }

        /// <summary>
        /// Physically copies the assembly to the specified Unity project.
        /// </summary>
        /// <param name="assemblyInfo">Path of the assembly to copy.</param>
        /// <param name="projectInfo">Path of the Unity project.</param>
        private void Copy(FileInfo assemblyInfo, DirectoryInfo projectInfo)
        {
            string assetsDir = Path.Combine(projectInfo.FullName, "Assets");
            string outDir = Path.Combine(assetsDir, AssemblyOutputDirectory);
            string filePath = Path.Combine(outDir, assemblyInfo.Name);

            if (!Directory.Exists(outDir))
                Directory.CreateDirectory(outDir);

            // TODO:  At this point we could add logic to make the copy process "smart". For instance
            //        we could decide if the assembly contains editor code on not. If assembly does,
            //        then we should copy the assembly into a sub directory named "Editor". 
            //
            //        We could also possibly utilize Mono.Cecil to convert any PDB files generated in
            //        Visual Studio into MDB's. 
            //
            //        These idea's were omitted from this example to keep things simple.


            assemblyInfo.CopyTo(filePath, true);
        }
    }
}
