﻿/*
 * Unite 2013: Custom Build Processes with Unity3D
 * =======================================================
 * Author: William Roberts - http://www.williamroberts.net
 * Date:   August 27, 2013
 * 
 * Schell Games - http://www.schellgames.com
 */

using System;
using System.IO;
using System.Reflection;
using Microsoft.Build.BuildEngine;

namespace UniteExample.Framework
{
    /// <summary>
    /// Represents the verbosity of the build log output.
    /// </summary>
    public enum BuildVerbosity
    {
        Quiet, 
        Minimal, 
        Normal, 
        Detailed, 
        Diagnostic 
    }


    /// <summary>
    /// Responsible for building the specifed MSBuild project and echoing the log output 
    /// to the command-line.
    /// </summary>
    public class BuildEngine : IDisposable
    {
        private Engine _engine;

        /// <summary>
        /// Determines the verbosity of the log file.
        /// </summary>
        public BuildVerbosity Verbosity
        {
            get;
            set;
        }


        public BuildEngine()
        {
            Verbosity = BuildVerbosity.Normal;

            // In this case we register the built-in "ConsoleLogger" as our logger. You can
            // also create a custom logger by inheriting Microsoft.Build.Framework.ILogger
            // as demonstrated in the sessions Power Point slide deck.
            _engine = new Engine();
            _engine.RegisterLogger(new ConsoleLogger());
        }

        /// <summary>
        /// Builds the specified MSBuild project.
        /// </summary>
        /// <param name="projectFile">THe MSBuild project file to build.</param>
        /// <returns>True if no errors were reported.</returns>
        /// <exception cref="InvalidOperationException">Thrown if the Execute method is invoked after the Dispose method was previously called.</exception>
        /// <exception cref="FileNotFoundException">Thrown if the project file does not exist.</exception>
        public bool Execute(string projectFile)
        {
            BuildPropertyGroup globalProperties;

            if (_engine == null)
                throw new InvalidOperationException("The BuildEngine was disposed!");

            if (!File.Exists(projectFile))
                throw new FileNotFoundException("Unable to open the specified project file!", projectFile);

            globalProperties = new BuildPropertyGroup();
            globalProperties.SetProperty("Verbosity", Verbosity.ToString());

            // The MSBuild engine will call the "Build" target that is specified in the "DemoProject.build" project.
            return _engine.BuildProjectFile(projectFile, new string[] { "Build" }, globalProperties);
        }

        /// <summary>
        /// Disposes the underlying MSBuild engine and ensures all file handles within the loggers are released.
        /// </summary>
        public void Dispose()
        {
            if (_engine != null)
            {
                // The Mono version of Microsoft.Build.BuildEngine.Engine does not contain a definition for the Shutdown method. 
                // Therefore, we must dynamically test for the method's existance prior to invoking it.
                MethodInfo shutDownMethod = _engine.GetType().GetMethod("Shutdown");

                if (shutDownMethod == null) // Mono
                {
                    // Unregistering the loggers ensures that ILogger's Shutdown method is invoked. Some Logger's require this
                    // in order to release file handles.
                    _engine.UnregisterAllLoggers();
                }
                else // Windows
                {
                    // The UnregisterAllLoggers method will be invoked by the Shutdown method.
                    shutDownMethod.Invoke(_engine, null);
                }

                _engine = null;
            }
        }
    }
}